<?php
/**
 * Elementor Widget
 * @package Appside
 * @since 1.0.0
 */

namespace Elementor;
class Panthar_List_Item_Widget extends Widget_Base
{

    /**
     * Get widget name.
     *
     * Retrieve Elementor widget name.
     *
     * @return string Widget name.
     * @since 1.0.0
     * @access public
     *
     */
    public function get_name()
    {
        return 'panthar-list-item-widget';
    }

    /**
     * Get widget title.
     *
     * Retrieve Elementor widget title.
     *
     * @return string Widget title.
     * @since 1.0.0
     * @access public
     *
     */
    public function get_title()
    {
        return esc_html__('List Item', 'panthar-core');
    }

    /**
     * Get widget icon.
     *
     * Retrieve Elementor widget icon.
     *
     * @return string Widget icon.
     * @since 1.0.0
     * @access public
     *
     */
    public function get_icon()
    {
        return 'eicon-post-slider';
    }

    /**
     * Get widget categories.
     *
     * Retrieve the list of categories the Elementor widget belongs to.
     *
     * @return array Widget categories.
     * @since 1.0.0
     * @access public
     *
     */
    public function get_categories()
    {
        return ['panthar_widgets'];
    }

    /**
     * Register Elementor widget controls.
     *
     * Adds different input fields to allow the user to change and customize the widget settings.
     *
     * @since 1.0.0
     * @access protected
     */
    protected function register_controls()
    {

        $this->start_controls_section(
            'settings_section',
            [
                'label' => esc_html__('General Settings', 'panthar-core'),
                'tab' => Controls_Manager::TAB_CONTENT,
            ]
        );

        $repeater = new Repeater();
        $repeater->add_control(
            'list_item_style',
            [
                'label' => esc_html__( 'list item style', 'panthar-core' ),
                'type' => Controls_Manager::SELECT,
                'default' => 'default-style',
                // 'separator' => 'after',
                'options' => [
                    'default-style' => esc_html__('Default', 'panthar-core'),
                    'dot-style' => esc_html__('Dot', 'panthar-core'),
                    'animate-dot-style' => esc_html__('Animation Dot', 'panthar-core'),
                ],
            ]
        );

        $repeater->add_control(
            'icon_selector',
            [
                'label' => esc_html__('Icon Selector', 'panthar-core'),
                'type' => Controls_Manager::SELECT,
                'default' => 'icon',
                'options' => [
                    'icon' => esc_html__('Icon', 'panthar-core'),
                    'text_icon' => esc_html__('Text Icon', 'panthar-core'),
                ],
                'condition' => [
                    'list_item_style' => 'default-style',
                ],
            ]
        );

        //list item icon or text field
        $repeater->add_control(
            'icon',
            [
                'label' => esc_html__('Icon', 'panthar-core'),
                'type' => Controls_Manager::ICONS,
                'description' => esc_html__('select Icon.', 'panthar-core'),
                'default' => [
                    'value' => 'fas fa-phone-alt',
                    'library' => 'solid',
                ],
                'condition' => [
                    'icon_selector' => 'icon',
                    'list_item_style' => 'default-style',
                ],
            ]
        );
        $repeater->add_control(
            'text_icon',
            [
                'label' => esc_html__('Text Icon', 'panthar-core'),
                'type' => Controls_Manager::TEXT,
                'description' => esc_html__('enter text.', 'panthar-core'),
                'default' => esc_html__('1', 'panthar-core'),
                'condition' => [
                    'icon_selector' => 'text_icon',
                    'list_item_style' => 'default-style',
                ],
            ]
        );

        //list item dot field
        $repeater->add_control(
            'dote_hide',
            [
                'label' => __('Dot Hide', 'panthar-core'),
                'type' => Controls_Manager::SWITCHER,
                'label_on' => __('Show', 'panthar-core'),
                'label_off' => __('Hide', 'panthar-core'),
                'return_value' => 'yes',
                'condition' => [
                    'list_item_style' => 'dot-style',
                ],
            ]
        );
        $repeater->add_control(
            'line_hide',
            [
                'label' => __('Line Hide', 'panthar-core'),
                'type' => Controls_Manager::SWITCHER,
                'label_on' => __('Show', 'panthar-core'),
                'label_off' => __('Hide', 'panthar-core'),
                'return_value' => 'yes',
                'condition' => [
                    'list_item_style' => 'dot-style',
                ],
            ]
        );

        //list item text field
        $repeater->add_control(
            'list_text', [
                'label' => esc_html__('List content', 'panthar-core'),
                'type' => Controls_Manager::TEXT,
                'default' => esc_html__('Security Consulting', 'panthar-core'),
                'label_block' => true,
            ]
        );

        //list item text field
        $repeater->add_control(
            'list_content', [
                'label' => esc_html__('List content', 'panthar-core'),
                'type' => Controls_Manager::TEXTAREA,
                'label_block' => true,
            ]
        );
        $this->add_control('list_item', [
            'label' => esc_html__('List Item', 'panthar-core'),
            'type' => Controls_Manager::REPEATER,
            'fields' => $repeater->get_controls(),
            'default' => [
                [
                    'text' => esc_html__( 'List Item #1', 'elementor' ),
                    'selected_icon' => [
                        'value' => 'fas fa-check',
                        'library' => 'fa-solid',
                    ],
                ],
            ],
        ]);
        $this->end_controls_section();

        $this->start_controls_section(
            'details_styling_settings_section',
            [
                'label' => esc_html__('Menu Details Styling Settings', 'panthar-core'),
                'tab' => Controls_Manager::TAB_STYLE,
            ]
        );
        $this->add_control('details_title_color', [
            'label' => esc_html__('Title Color', 'panthar-core'),
            'type' => Controls_Manager::COLOR,
            'selectors' => [
                "{{WRAPPER}} .list-item-wrap ul li .content" => "color: {{VALUE}}",
                "{{WRAPPER}} .list-item-wrap ul li .style-dot" => "background: {{VALUE}}",
                "{{WRAPPER}} .list-item-wrap ul li .style-bar" => "background: {{VALUE}}",
            ]
        ]);
        $this->add_control('details_dot_color', [
            'label' => esc_html__('Title Dot Color', 'panthar-core'),
            'type' => Controls_Manager::COLOR,
            'selectors' => [
                "{{WRAPPER}} .list-item-wrap ul li .style-dot" => "background: {{VALUE}}",
            ]
        ]);
        $this->add_control('details_bar_color', [
            'label' => esc_html__('Title Bar Color', 'panthar-core'),
            'type' => Controls_Manager::COLOR,
            'selectors' => [
                "{{WRAPPER}} .list-item-wrap ul li .style-bar" => "background: {{VALUE}}",
            ]
        ]);
        $this->end_controls_section();
    }


    /**
     * Render Elementor widget output on the frontend.
     * Written in PHP and used to generate the final HTML.
     *
     * @since 1.0.0
     * @access protected
     */
    protected function render()
    {
        $settings = $this->get_settings_for_display();
        ?>
        <div class="content-wrapper">
            <div class="list-item-wrap">
                <ul>
                    <?php foreach ($settings['list_item'] as $item) : ?>

                        <!-- list left icon style -->
                        <?php
                            $dote_hide_none = '';
                            $line_hide_none = '';
                            if($item['dote_hide'] == 'yes'):
                                $dote_hide_none = 'd-none';  
                            endif;
                            if($item['line_hide'] == 'yes'):
                                $line_hide_none = 'd-none';  
                            endif;

                            if($item['list_item_style'] == 'dot-style'):
                                $list_item_style = '<span class="style-dot '.$dote_hide_none.'"></span> <span class="style-bar '.$line_hide_none.'"></span>';
                            elseif($item['list_item_style'] == 'animate-dot-style'):
                                $list_item_style = '<span class="style-animate-dot"></span>';
                            endif;
                        ?>

                        <li>
                            <?php if(!empty($list_item_style)) : 
                                echo $list_item_style;
                            else : ?>
                                <div class="icon-wrap d-inline-block float-left">
                                    <?php if ($item['icon_selector'] == 'icon'): ?>
                                        <span class="icon">
                                            <?php
                                            Icons_Manager::render_icon($item['icon'], ['aria-hidden' => 'true']);
                                            ?>
                                        </span>
                                    <?php else:
                                        printf('<div class="text-icon d-inline-block"><span>%1$s</span></div>', esc_html($item['text_icon']));
                                    endif; ?>
                                </div>
                            <?php endif; ?>
                            <div class="content d-inline-block"><?php echo esc_html($item['list_text']) ?> </div>
                            <span class="detail-content"> <?php echo esc_html($item['list_content']) ?></span>
                        </li>
                        
                    <?php endforeach; ?>
                </ul>
            </div>
        </div>
        <?php
    }
}

Plugin::instance()->widgets_manager->register_widget_type(new Panthar_List_Item_Widget());